import 'dart:convert';
import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:http/http.dart' as http;
import 'package:search_choices/search_choices.dart';
import 'package:shared_preferences/shared_preferences.dart';

import '../../../Networking/ApiService.dart';
import '../../../Networking/DialogHelper.dart';
import '../../../Networking/getAuthHeaders.dart';
import '../../Admin/AdminDashboardScreen.dart';
import '../../Doctor/DoctorDashboardScreen.dart';
import '../../Hospital/HospitalDashboardScreen.dart';
import '../../Transcrpit/TranscrpitDashboardScreen.dart';

class AdduserPage extends StatefulWidget {
  final String result;

  AdduserPage({
    required this.result,
  });

  @override
  State<AdduserPage> createState() => _AdduserPageState();
}

class _AdduserPageState extends State<AdduserPage> {
  final ApiService apiService = ApiService();
  late SharedPreferences sharedPreferences;
  String Firstname = "",
      USERID = "",
      ROLEID = "",
      Pendingfile = "",
      Recivedfile = "";
  final _formKey = GlobalKey<FormState>();
  final TextEditingController _nameController = TextEditingController();
  final TextEditingController _mobileController = TextEditingController();
  final TextEditingController _emailController = TextEditingController();
  final TextEditingController _expController = TextEditingController();
  final TextEditingController _passwordController = TextEditingController();
  final TextEditingController _confirmPasswordController =
      TextEditingController();
  String? selecthospital, HOSPITALID, HOSPITALNAME;
  Future<List<Map<String, dynamic>>>? _hospitalFuture;
  bool _obscurePassword = true;
  bool _obscureConfirmPassword = true;
  String? _hospitalError;
  final FocusNode _nameFocus = FocusNode();
  final FocusNode _mobileFocus = FocusNode();
  final FocusNode _emailFocus = FocusNode();
  final FocusNode _expFocus = FocusNode();
  final FocusNode _passwordFocus = FocusNode();
  final FocusNode _confirmPasswordFocus = FocusNode();
  @override
  void initState() {
    super.initState();

    initialeclass();
    if (widget.result == "doctor") {
      _hospitalFuture = APIHOSPITALlist(); // Call your API method
    }
  }

  initialeclass() async {
    sharedPreferences = await SharedPreferences.getInstance();
    setState(() {
      ROLEID = sharedPreferences.getString("userRole") ?? '';
      Firstname = sharedPreferences.getString("username") ?? '';
      USERID = sharedPreferences.getString("user_id") ?? '';
    });

    // ROLENAME = sharedPreferences.getString("role_name") ?? '';
  }

  Future<void> _saveDoctor() async {
    bool isValid = _formKey.currentState!.validate();

    // ✅ Hospital validation (only for doctors)
    if (widget.result == "doctor" && (HOSPITALID == null || HOSPITALID!.isEmpty)) {
      setState(() {
        _hospitalError = "Please select a hospital";
      });
      isValid = false;
    } else {
      setState(() {
        _hospitalError = null; // clear hospital error
      });
    }

    // ✅ Experience validation (doctor, quality, transcript)
    if ((widget.result == "doctor" || widget.result == "quality" || widget.result == "transcript") &&
        (_expController.text.isEmpty)) {
      // ScaffoldMessenger.of(context).showSnackBar(
      //   SnackBar(content: Text("Please enter experience")),
      // );
      isValid = false;
    }

    if (!isValid) return;

    DialogHelper.showLoading(context);

    try {
      final headers = await getAuthHeaders();

      // 🔹 Map for role IDs
      final Map<String, int> roleMap = {
        "hospital": 2,
        "doctor": 4,
        "quality": 11,
        "transcript": 5,
      };

      int? roleId = roleMap[widget.result.toLowerCase()];

      // 🔹 Common payload
      Map<String, dynamic> requestData = {
        "name": _nameController.text.trim(),
        "email": _emailController.text.trim(),
        "mobile": _mobileController.text.trim(),
        "password": _passwordController.text.trim(),
        "password_confirmation": _confirmPasswordController.text.trim(),
        "roleId": roleId,
      };

      // 🔹 Add hospitalId only if doctor
      if (widget.result == "doctor") {
        requestData["hospitalId"] = HOSPITALID; // from dropdown
      }

      // 🔹 Add experience if doctor, quality, transcript
      if (widget.result == "quality" || widget.result == "transcript" || widget.result == "doctor") {
        requestData["experience"] = int.tryParse(_expController.text) ?? 0;
      }
      final url = "https://app.capscribes.com/api/users";

// 🔹 Print URL and request data for debugging
      print("POST URL: $url");
      print("Request Data: ${jsonEncode(requestData)}");
      print("Headers: $headers");

      final response = await http.post(
        Uri.parse(url),
        headers: headers,
        body: jsonEncode(requestData),
      );

      final responseData = json.decode(response.body);
      if (response.statusCode == 200 || response.statusCode == 201) {
        DialogHelper.dismissDialog();
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text("${widget.result} Added Successfully!")),
        );
        // ✅ Unfocus all fields to hide keyboard
        FocusScope.of(context).unfocus();

        // ✅ Reset all fields and variables
        _formKey.currentState?.reset();
        _nameController.clear();
        _mobileController.clear();
        _emailController.clear();
        _expController.clear();
        _passwordController.clear();
        _confirmPasswordController.clear();
        setState(() {
          selecthospital = null;
          HOSPITALID = null;
          HOSPITALNAME = null;
          _hospitalError = null;
        });

        // ✅ Unfocus all fields after a short delay to ensure UI updates
        Future.delayed(Duration(milliseconds: 100), () {
          _nameFocus.unfocus();
          _mobileFocus.unfocus();
          _emailFocus.unfocus();
          _expFocus.unfocus();
          _passwordFocus.unfocus();
          _confirmPasswordFocus.unfocus();
        });

      } else {
        DialogHelper.dismissDialog();

        String errorMessage = responseData['message'] ?? response.body;
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text("$errorMessage")),
        );

      }
    } catch (e) {
      DialogHelper.dismissDialog();
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text("Error: $e")),
      );
    } finally {
      DialogHelper.dismissDialog();
    }
  }


  Color getBackgroundColor(String roleId) {
    switch (roleId) {
      case "2": // Admin
        return Color(0xffC85D06);

      case "3": // Admin
        return Color(0xffb3b300);
      case "4": // Doctor
        return Color(0xff1F5C93); // Greenish
      case "5": // Hospital
        return Color(0xff004D40); // Purple
      default:
        return Colors.grey; // Default or unknown
    }
  }

  /// Dynamic Title
  String getPageTitle() {
    switch (widget.result) {
      case "doctor":
        return "Add Doctor";
      case "hospital":
        return "Add Hospital";
      case "transcript":
        return "Add Transcript";
      case "quality":
        return "Add Quality";
      default:
        return "Add User";
    }
  }

  /// Dynamic First Label
  String getNameLabel() {
    switch (widget.result) {
      case "doctor":
        return "Doctor Name";
      case "hospital":
        return "Hospital Name";
      case "transcript":
        return "Transcript Name";
      case "quality":
        return "Quality Name";
      default:
        return "Name";
    }
  }

  Future<List<Map<String, dynamic>>> APIHOSPITALlist() async {
    try {
      final headers = await getAuthHeaders();
      final response = await http.get(
          Uri.parse(apiService.BaseURL + apiService.HOSPITALLIST),
          headers: headers);
      print(apiService.BaseURL + apiService.HOSPITALLIST);

      if (response.statusCode == 200) {
        // Decode the JSON response
        final decoded = json.decode(response.body);

        // Access the list inside the "data" key
        final List<dynamic> data = decoded['hospitals'];

        // Convert and return as List<Map<String, dynamic>>
        return List<Map<String, dynamic>>.from(data);
      } else {
        throw Exception('Failed to load doctor list');
      }
    } on SocketException {
      Fluttertoast.showToast(
        msg: "Network error! Please retry.",
        toastLength: Toast.LENGTH_SHORT,
        gravity: ToastGravity.BOTTOM,
        backgroundColor: Colors.red,
        textColor: Colors.white,
      );
      throw Exception('Network error: Unable to fetch');
    } catch (e) {
      print("Error: $e");
      throw Exception('An error occurred: $e');
    }
  }

  @override
  Widget build(BuildContext context) {
    return WillPopScope(
      onWillPop: () async {
        switch (ROLEID) {
          case "2":
            Navigator.pushReplacement(
              context,
              MaterialPageRoute(builder: (context) => AdminDashboardScreen()),
            );
            break;
          case "3":
            Navigator.pushReplacement(
              context,
              MaterialPageRoute(
                  builder: (context) => TranscrpitDashboardScreen()),
            );
            break;
          case "4":
            Navigator.pushReplacement(
              context,
              MaterialPageRoute(builder: (context) => DoctorDashboardScreen()),
            );
            break;
          case "5":
            Navigator.pushReplacement(
              context,
              MaterialPageRoute(
                  builder: (context) => HospitalDashboardScreen()),
            );
            break;
          default:
            // Handle unknown ROLEID or do nothing
            break;
        }

        return false; // Prevent default pop
      },
      child: Scaffold(
        appBar: AppBar(
          title: Text(
            getPageTitle(),
            style: TextStyle(color: Colors.white),
          ),
          backgroundColor: getBackgroundColor(ROLEID),
          iconTheme: IconThemeData(color: Colors.white),
        ),
        body: SingleChildScrollView(
          padding: EdgeInsets.all(16),
          child: Form(
            // ✅ wrap with Form
            key: _formKey, // ✅ connect form key here
            child: Column(
              children: [
                SizedBox(height: 30),
                // Name
                TextFormField(
                  controller: _nameController,
                  focusNode: _nameFocus,
                  decoration: InputDecoration(
                    labelText: getNameLabel(),
                    prefixIcon: Icon(Icons.person),
                    border: OutlineInputBorder(),
                  ),
                  validator: (value) =>
                      value!.isEmpty ? "Please enter ${getNameLabel()}" : null,
                ),

                // ✅ Show Hospital Dropdown only if "doctor" is selected


        const SizedBox(height: 30),

                // Mobile
                TextFormField(
                  controller: _mobileController,
                  focusNode: _mobileFocus,
                  keyboardType: TextInputType.phone,
                  inputFormatters: [
                    FilteringTextInputFormatter.digitsOnly,             // Allow only digits
                    LengthLimitingTextInputFormatter(10),               // Limit to 10 digits
                  ],
                  decoration: const InputDecoration(
                    labelText: "Mobile Number",
                    prefixIcon: Icon(Icons.phone),
                    border: OutlineInputBorder(),
                  ),
                  validator: (value) {
                    if (value!.isEmpty) return "Enter mobile number";
                    if (value.length != 10)
                      return "Enter valid 10-digit number";
                    return null;
                  },
                ),
                const SizedBox(height: 30),

                // Email
                TextFormField(
                  controller: _emailController,
                  focusNode: _emailFocus,
                  keyboardType: TextInputType.emailAddress,
                  decoration: const InputDecoration(
                    labelText: "Email",
                    prefixIcon: Icon(Icons.email),
                    border: OutlineInputBorder(),
                  ),
                  validator: (value) {
                    if (value!.isEmpty) return "Enter email";
                    if (!RegExp(r'^[^@]+@[^@]+\.[^@]+').hasMatch(value)) {
                      return "Enter valid email";
                    }
                    return null;
                  },
                ),
                const SizedBox(height: 30),

                // Experience
                if (widget.result == "doctor" ||
                    widget.result == "transcript" ||
                    widget.result == "quality") ...[
                  TextFormField(
                    controller: _expController,
                    focusNode: _expFocus,
                    keyboardType: TextInputType.number,
                    decoration: const InputDecoration(
                      labelText: "Experience (Years)",
                      prefixIcon: Icon(Icons.work),
                      border: OutlineInputBorder(),
                    ),
                    validator: (value) {
                      if ((widget.result == "doctor" ||
                          widget.result == "transcript" ||
                          widget.result == "quality") &&
                          (value == null || value.isEmpty)) {
                        return "Enter experience";
                      }
                      return null;
                    },
                  ),
                  const SizedBox(height: 30),
                ],

                if (widget.result == "doctor")
                  Column(
                    children: [
                      Padding(
                        padding: const EdgeInsets.symmetric(vertical: 8),
                        child: Container(
                          decoration: BoxDecoration(
                            border: Border.all(color: Colors.black), // border color
                            borderRadius: BorderRadius.circular(5), // rounded corners
                          ),
                          child: Padding(
                            padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 4),
                            child: Row(
                              children: [
                                Icon(Icons.local_hospital, color: Colors.black), // hospital icon
                                SizedBox(width: 8),
                                Expanded(
                                  child: FutureBuilder<List<Map<String, dynamic>>>(
                                    future: _hospitalFuture,
                                    builder: (context, snapshot) {
                                      if (snapshot.connectionState == ConnectionState.waiting) {
                                        return SizedBox(
                                            height: 5,
                                            width: 5,
                                            child: CircularProgressIndicator());
                                      } else if (snapshot.hasError || !snapshot.hasData) {
                                        return Text('No Hospital Found',
                                            style: TextStyle(color: Colors.grey));
                                      } else {
                                        return SearchChoices.single(
                                          padding: EdgeInsets.zero,
                                          items: snapshot.data!
                                              .map<DropdownMenuItem<String>>((item) {
                                            return DropdownMenuItem<String>(
                                              value: '${item['hospital_name']},${item['userId']}',
                                              child: Text(
                                                item['hospital_name'],
                                                style: TextStyle(fontSize: 14),
                                              ),
                                            );
                                          }).toList(),
                                          value: selecthospital,
                                          hint: Text(
                                            "Select Hospital",
                                            style: TextStyle(
                                                color: Colors.black,
                                                fontSize: 14,
                                                fontWeight: FontWeight.bold),
                                          ),
                                          searchHint: "Select Hospital",
                                          underline: Container(),
                                          icon: Icon(Icons.arrow_drop_down, color: Colors.grey),
                                          onChanged: (newValue) {
                                            setState(() {
                                              selecthospital = newValue as String;
                                              List<String> parts = newValue.split(',');
                                              HOSPITALID = parts[1];
                                              HOSPITALNAME = parts[0];
                                            });
                                          },
                                          closeButton: "Close",
                                          isExpanded: true,
                                          displayClearIcon: false,
                                          style: TextStyle(color: Colors.black),
                                          dialogBox: true,
                                        );
                                      }
                                    },
                                  ),
                                ),
                              ],
                            ),
                          ),
                        ),
                      ),
                      // Show error text below
                      if (_hospitalError != null)
                        Padding(
                          padding: const EdgeInsets.only(left: 12, top: 4),
                          child: Align(
                            alignment: Alignment.centerLeft, // force left alignment
                            child: Text(
                              _hospitalError!,
                              style: TextStyle(color: Colors.redAccent, fontSize: 12),
                            ),
                          ),
                        ),


                      const SizedBox(height: 30),
                    ],
                  ),
                // Password
                TextFormField(
                  controller: _passwordController,
                  obscureText: _obscurePassword,
                  focusNode: _passwordFocus ,
                  decoration: InputDecoration(
                    labelText: "Password",
                    prefixIcon: const Icon(Icons.lock),
                    border: const OutlineInputBorder(),
                    suffixIcon: IconButton(
                      icon: Icon(_obscurePassword
                          ? Icons.visibility_off
                          : Icons.visibility),
                      onPressed: () {
                        setState(() {
                          _obscurePassword = !_obscurePassword;
                        });
                      },
                    ),
                  ),
                  validator: (value) {
                    if (value!.isEmpty) return "Enter password";
                    if (value.length < 6)
                      return "Password must be at least 6 characters";
                    return null;
                  },
                ),
                const SizedBox(height: 30),

                // Confirm Password
                TextFormField(
                  controller: _confirmPasswordController,
                  obscureText: _obscureConfirmPassword,
                  focusNode: _confirmPasswordFocus  ,
                  decoration: InputDecoration(
                    labelText: "Confirm Password",
                    prefixIcon: const Icon(Icons.lock_outline),
                    border: const OutlineInputBorder(),
                    suffixIcon: IconButton(
                      icon: Icon(_obscureConfirmPassword
                          ? Icons.visibility_off
                          : Icons.visibility),
                      onPressed: () {
                        setState(() {
                          _obscureConfirmPassword = !_obscureConfirmPassword;
                        });
                      },
                    ),
                  ),
                  validator: (value) {
                    if (value!.isEmpty) return "Re-enter password";
                    if (value != _passwordController.text) {
                      return "Passwords do not match";
                    }
                    return null;
                  },
                ),
                const SizedBox(height: 30),

                // Save Button
                SizedBox(
                  width: double.infinity,
                  child: ElevatedButton(
                    onPressed: _saveDoctor,
                    style: ElevatedButton.styleFrom(
                      backgroundColor: Color(0xffFF9C4B),
                      padding: const EdgeInsets.symmetric(vertical: 14),
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(8),
                      ),
                    ),
                    child: const Text(
                      "Save",
                      style: TextStyle(
                          fontSize: 18,
                          fontWeight: FontWeight.bold,
                          color: Colors.white),
                    ),
                  ),
                )
              ],
            ),
          ),
        ),
      ),
    );
  }
}
