import 'dart:async';
import 'dart:io';

import 'package:audio_waveforms/audio_waveforms.dart';
import 'package:capscribes/UI/Admin/AdminDashboardScreen.dart';
import 'package:capscribes/UI/Admin/AdminIntroScreen.dart';
import 'package:capscribes/UI/Doctor/DoctorDashboardScreen.dart';
import 'package:capscribes/UI/Hospital/HospitalDashboardScreen.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_svg/svg.dart';
import 'package:http/http.dart' as http;
import 'package:path_provider/path_provider.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:sliding_action_button/sliding_action_button.dart';
import 'package:flutter_sound/flutter_sound.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:path/path.dart' as p;
import '../../../Networking/ApiService.dart';
import '../../../Networking/DialogHelper.dart';
import '../../../Networking/getAuthHeaders.dart';
import '../DoctorIntroScreen.dart';
import '../../Hospital/HospitalIntroScreen.dart';

class VoiceRecorderPage extends StatefulWidget {
  @override
  _VoiceRecorderPageState createState() => _VoiceRecorderPageState();
}

class _VoiceRecorderPageState extends State<VoiceRecorderPage> {

  final ApiService apiService = ApiService();
  late SharedPreferences sharedPreferences;
  bool _isRecording = false;
  bool _isRecorded = false;
  bool _isPlaying = false;
  bool _showUploadUI = false;

  String? _filePath;
  String _recordDuration = "00:00";
  Timer? _timer;
  int _seconds = 0;

  Timer? _playbackTimer;
  int _playbackSeconds = 0;
  String _playbackDuration = "00:00";
  bool _isPaused = false;
  late RecorderController _recorderController;
  late PlayerController _playerController;
  final TextEditingController _audioNameController = TextEditingController();
  String Firstname="",USERID="",ROLEID="";
  @override
  void initState() {
    super.initState();
    _recorderController = RecorderController();
    _playerController = PlayerController();
    _requestPermission();
    initialeclass();
  }


  initialeclass() async {
    sharedPreferences = await SharedPreferences.getInstance();
    Firstname = sharedPreferences.getString("username") ?? '';
    USERID = sharedPreferences.getString("user_id") ?? '';
    ROLEID = sharedPreferences.getString("userRole") ?? '';

  }



  Future<void> _requestPermission() async {
    await Permission.microphone.request();
    await Permission.storage.request();
  }

  Future<String> _getFilePath() async {
    final dir = await getApplicationDocumentsDirectory();
    return "${dir.path}/audio_${DateTime.now().millisecondsSinceEpoch}.aac";
  }

  void _startRecording() async {
    _filePath = await _getFilePath();
    await _recorderController.record(path: _filePath!);

    _seconds = 0;
    _timer = Timer.periodic(Duration(seconds: 1), (timer) {
      setState(() {
        _seconds++;
        final minutes = (_seconds ~/ 60).toString().padLeft(2, '0');
        final seconds = (_seconds % 60).toString().padLeft(2, '0');
        _recordDuration = "$minutes:$seconds";
      });
    });

    setState(() {
      _isRecording = true;
      _isRecorded = false;
      _showUploadUI = false;
    });
  }

  void _stopRecording() async {
    await _recorderController.stop();
    _timer?.cancel();

    setState(() {
      _isRecording = false;
      _isRecorded = true;
      _showUploadUI = true;
    });
  }

  void _playRecording() async {
    if (_filePath != null && File(_filePath!).existsSync()) {
      await _playerController.stopPlayer();
      await _playerController.preparePlayer(path: _filePath!);
      await _playerController.startPlayer();

      setState(() {
        _isPlaying = true;
        _playbackSeconds = 0;
        _playbackDuration = "00:00";
      });

      _playbackTimer?.cancel();
      _playbackTimer = Timer.periodic(Duration(seconds: 1), (timer) {
        setState(() {
          _playbackSeconds++;
          _playbackDuration = _formatDuration(_playbackSeconds);
        });
      });

      _playerController.onCompletion.listen((event) {
        setState(() {
          _isPlaying = false;
          _playbackSeconds = 0;
          _playbackDuration = "00:00";
        });
        _playbackTimer?.cancel();
      });
    }
  }


  String _formatDuration(int seconds) {
    final minutes = (seconds ~/ 60).toString().padLeft(2, '0');
    final secs = (seconds % 60).toString().padLeft(2, '0');
    return "$minutes:$secs";
  }

  Future<void> _uploadAudio() async {

    String name = _audioNameController.text.trim();

    if (_filePath == null || !_isRecorded) {
      ScaffoldMessenger.of(context)
          .showSnackBar(SnackBar(content: Text('No recorded file found')));
      return;
    }

    if (name.isEmpty) {
      ScaffoldMessenger.of(context)
          .showSnackBar(SnackBar(content: Text('Please enter audio name')));
      return;
    }

    try {

      DialogHelper.showLoading(context);
      // Replace spaces with underscores
      String cleanedFileName = name.replaceAll(' ', '_') + p.extension(_filePath!);
      print("=== Uploading Audio ===");
      print("Audio Name: $name");
      print("Cleaned File Name: $cleanedFileName");
      print("File Path: $_filePath");
      print("User ID: $USERID");
      print("API Endpoint: ${apiService.BaseURL + apiService.VOICEUPLOAD}");

      final headers = await getAuthHeaders();
      var request = http.MultipartRequest(
        'POST',
        Uri.parse(apiService.BaseURL + apiService.VOICEUPLOAD ), // Replace with your endpoint
      );

      request.headers.addAll(headers);
      request.files.add(
        await http.MultipartFile.fromPath(
          'medical_audio',
          _filePath!,
          filename: cleanedFileName,
        ),
      );
      request.fields['user_id'] = USERID;
      request.fields['file_name'] = name;// Replace with actual user ID

      var response = await request.send();

      if (response.statusCode == 200) {
        DialogHelper.dismissDialog();
        bool? uploaded = await showDialog(
          context: context,
          barrierDismissible: false,
          builder: (BuildContext context) {
            Future.delayed(Duration(seconds: 3), () {
              if (Navigator.canPop(context)) {
                Navigator.pop(context, true); // Return true to calling code
              }
            });

            return WillPopScope(
              onWillPop: () async => false,
              child: AlertDialog(
                insetPadding: EdgeInsets.zero,
                contentPadding: EdgeInsets.zero,
                shape: RoundedRectangleBorder(
                  borderRadius: BorderRadius.circular(16),
                ),
                content: Container(
                  decoration: BoxDecoration(
                    color: Colors.green.shade50,
                    borderRadius: BorderRadius.circular(16),
                  ),
                  padding: EdgeInsets.all(20),
                  child: Column(
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      Icon(Icons.check_circle, color: Colors.green, size: 50),
                      SizedBox(height: 12),
                      Text(
                        'Audio Uploaded Successfully',
                        style: TextStyle(
                            fontSize: 18,
                            fontWeight: FontWeight.bold,
                            color: Colors.green),
                        textAlign: TextAlign.center,
                      ),
                    ],
                  ),
                ),
              ),
            );
          },
        );

        if (uploaded == true) {
          setState(() {
            // Reset UI to initial state
            _audioNameController.clear();
            _filePath = null;
            _isRecorded = false;
            _isRecording = false;
            _isPlaying = false;
            _isPaused = false;
            _recordDuration = "00:00";
            _playbackDuration = "00:00";
            _showUploadUI = false;

            // Optionally reset audio controllers
            _playerController.stopPlayer();
            _recorderController.reset();
          });
        }
      } else {
        DialogHelper.dismissDialog();
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Upload failed (Status: ${response.statusCode})')),
        );
      }
    } catch (e) {
      DialogHelper.dismissDialog();
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Upload error: $e')),
      );
    }
  }

  Widget _actionButton(IconData icon, Color color, VoidCallback? onPressed) {
    return GestureDetector(
      onTap: onPressed,
      child: Container(
        width: 60.0,  // Adjust size
        height: 60.0, // Adjust size
        decoration: BoxDecoration(
          color: color.withOpacity(0.2),  // Add background color with opacity
          shape: BoxShape.circle,         // Make it circular
        ),
        child: Icon(
          icon,
          color: color,
          size: 32,  // Adjust icon size
        ),
      ),
    );
  }


  @override
  void dispose() {
    _timer?.cancel();
    _recorderController.dispose();
    _playerController.dispose();
    _audioNameController.dispose();
    super.dispose();
  }
  void _resumeRecording() async {
    await _playerController.startPlayer(); // Resumes from where it paused
    setState(() {
      _isPlaying = true;
      _isPaused = false;
    });

    _playbackTimer?.cancel();
    _playbackTimer = Timer.periodic(Duration(seconds: 1), (timer) {
      setState(() {
        _playbackSeconds++;
        _playbackDuration = _formatDuration(_playbackSeconds);
      });
    });

    _playerController.onCompletion.listen((event) {
      setState(() {
        _isPlaying = false;
        _isPaused = false;
      });
      _playbackTimer?.cancel();
    });
  }
  void _pauseRecording() async {
    await _playerController.pausePlayer(); // Use pause if available, otherwise stop
    _playbackTimer?.cancel();
    setState(() {
      _isPlaying = false;
      _isPaused = true;
    });
  }
  @override
  Widget build(BuildContext context) {
    return
      PopScope(
        canPop: false, // prevent default back
        onPopInvoked: (didPop) async {
          if (!didPop) {
            // Replace current screen with another one
            Navigator.pushReplacement(
              context,
              MaterialPageRoute(builder: (context) => DoctorDashboardScreen()),
            );
          }
        },
      child: Scaffold(
        backgroundColor: Color(0xfff5f5f5),
        appBar: AppBar(
          backgroundColor:  Color(0xff1F5C93),
          leading: IconButton(
            icon: Icon(
              Icons.arrow_back_ios_new_outlined,
              color: Colors.white,
            ),
            onPressed: () {
              Navigator.pushReplacement(
                context,
                MaterialPageRoute(builder: (context) => DoctorDashboardScreen()),
              );
            },
          ),
          title: Text('Record', style: TextStyle(color: Colors.white)),
        ),
        body: Stack(
          children: [
            Container(
              height: 150,
              decoration: BoxDecoration(
                color:  Color(0xff1F5C93), // Example background color
                borderRadius: BorderRadius.only(
                  bottomLeft: Radius.circular(20.0),
                  bottomRight: Radius.circular(20.0),
                ),
              ), // Example background color
            ),
            SingleChildScrollView(
              child: Column(
                children: [
                  Container(
                    margin: EdgeInsets.all(16),
                    padding: EdgeInsets.symmetric(vertical: 40),
                    decoration: BoxDecoration(
                      color: Colors.white,
                      borderRadius: BorderRadius.circular(32),
                    ),
                    width: double.infinity,
                    child: Column(
                      children: [
                        Text(
                          (_isPlaying || _isPaused) ? _playbackDuration : _recordDuration,
                          style: TextStyle(fontSize: 36, fontWeight: FontWeight.bold),
                        ),

                        SizedBox(height: 20),
                        GestureDetector(
                          onTap: (_isRecording || _isPlaying) ? null : _startRecording,
                          child: CircleAvatar(
                            radius: 32,
                            backgroundColor: Color(0xff1a73e8).withOpacity(0.2),
                            child: Icon(
                              _isRecording ? Icons.mic_off : Icons.mic,
                              color: Color(0xff1a73e8),
                              size: 32,
                            ),
                          ),
                        ),
                        SizedBox(height: 20),
                        _isRecording
                            ? AudioWaveforms(
                          enableGesture: true,
                          size: Size(MediaQuery.of(context).size.width / 2, 50),
                          recorderController: _recorderController,
                          waveStyle: WaveStyle(
                            waveColor: Colors.blue,
                            extendWaveform: true,
                            showMiddleLine: false,
                            spacing: 8.0,
                            waveThickness: 5.0,
                            scaleFactor: 200,
                          ),
                          padding: const EdgeInsets.only(left: 18),
                          margin: const EdgeInsets.symmetric(horizontal: 15),
                        )
                            : (_isRecorded
                            ? AudioFileWaveforms(
                          playerController: _playerController,
                          size: Size(MediaQuery.of(context).size.width / 2, 50),
                          playerWaveStyle: PlayerWaveStyle(
                            liveWaveColor: Colors.blue,
                            fixedWaveColor: Colors.grey.shade300,
                            waveThickness: 4.0,
                          ),
                        )
                            : SvgPicture.asset(
                          'assets/wave_placeholder.svg',
                          height: 70,
                          fit: BoxFit.contain,
                        )
                        ),

                      ],
                    ),
                  ),

                  // Buttons
                  Padding(
                    padding: const EdgeInsets.all(16.0),
                    child: Container(
                      decoration: BoxDecoration(
                        color: Colors.white,
                        borderRadius: BorderRadius.circular(20),
                        boxShadow: [
                          BoxShadow(
                            color: Colors.black12,
                            blurRadius: 10,
                            offset: Offset(0, 4),
                          ),
                        ],
                      ),
                      padding: const EdgeInsets.symmetric(vertical: 20, horizontal: 16),
                      child:Row(
                        mainAxisAlignment: MainAxisAlignment.spaceAround,
                        children: [
                          // Record button (if not recording or playing)
                          if (!_isRecording && !_isPlaying)
                            _actionButton(Icons.mic, Colors.blue, _startRecording),

                          // Delete button
                          if (_isRecorded)
                            _actionButton(Icons.delete_outline, Colors.red, _isPlaying
                                ? null
                                : () {
                              if (_filePath != null) {
                                File(_filePath!).delete();
                                setState(() {
                                  _isRecorded = false;
                                  _filePath = null;
                                  _recordDuration = "00:00";
                                  _showUploadUI = false;
                                  _playbackDuration = "00:00";
                                });
                              }
                            }),

                          // Stop button
                          if (_isRecording)
                            _actionButton(Icons.stop, Colors.red, () {
                              _stopRecording();
                            }),

                          // Play / Pause / Resume button
                          if (_isRecorded)
                            _isPlaying
                                ? _actionButton(Icons.pause, Color(0xff1a73e8), _pauseRecording)
                                : _isPaused
                                ? _actionButton(Icons.play_arrow, Colors.green, _resumeRecording)
                                : _actionButton(Icons.play_arrow, Colors.green, _playRecording),
                        ],
                      )

                    ),
                  ),


                  // Upload UI
                  if (_showUploadUI)
                    Padding(
                      padding: const EdgeInsets.all(16.0),
                      child: Column(
                        children: [
                          TextField(
                            controller: _audioNameController,
                            decoration: InputDecoration(
                              labelText: 'Audio Name',
                              border: OutlineInputBorder(),
                            ),
                          ),
                          SizedBox(height: 10),
                          ElevatedButton.icon(
                            onPressed: _isPlaying ? null : _uploadAudio,
                            icon: Icon(Icons.cloud_upload,color: Colors.white,),
                            label: Text('Upload Audio',style: TextStyle(color: Colors.white),),
                            style: ElevatedButton.styleFrom(
                              backgroundColor: Colors.green,
                              shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                            ),
                          ),
                        ],
                      ),
                    ),
                ],
              ),
            ),
          ],
        ),
      ),
    );
  }
}
