import 'dart:convert';
import 'dart:io';

import 'package:capscribes/UI/Admin/AdminDashboardScreen.dart';
import 'package:capscribes/UI/Doctor/DoctorDashboardScreen.dart';
import 'package:capscribes/UI/Hospital/HospitalDashboardScreen.dart';
import 'package:capscribes/UI/Hospital/dashboard/HospitalSingleDoctorProfileScreen.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:http/http.dart' as http;
import 'package:shared_preferences/shared_preferences.dart';

import '../../../Networking/ApiService.dart';
import '../../../Networking/getAuthHeaders.dart';
import '../../Hospital/dashboard/AllDoctorListsScreen.dart';
import '../../Hospital/dashboard/SelectedDoctorFileScreen.dart';
import 'AdminSingleDoctorProfileScreen.dart';


class AdminAllDoctorListsScreen extends StatefulWidget {
  const AdminAllDoctorListsScreen({super.key});

  @override
  State<AdminAllDoctorListsScreen> createState() => _AdminAllDoctorListsScreenState();
}


class _AdminAllDoctorListsScreenState extends State<AdminAllDoctorListsScreen> {
  final ApiService apiService = ApiService();
  late SharedPreferences sharedPreferences;
  String Firstname="",USERID="",ROLEID="",selectdoctor="";


  Future<List<Doctor>>? _doctorFuture;
  void initState() {
    super.initState();
    initialeclass();
  }

  // Initialize class and fetch doctor data
  initialeclass() async {
    sharedPreferences = await SharedPreferences.getInstance();
    Firstname = sharedPreferences.getString("username") ?? '';
    USERID = sharedPreferences.getString("user_id") ?? '';
    ROLEID = sharedPreferences.getString("userRole") ?? '';

    setState(() {
      _doctorFuture = fetchDoctors();
    });
  }

  // Fetch doctor data from API
  Future<List<Doctor>>  fetchDoctors() async {
    try {
      final headers = await getAuthHeaders();

      final url = Uri.parse(apiService.BaseURL + apiService.Doctorlist + "/");

      final response = await http.get(url, headers: headers);

      print("Request URL: $url");
      print("Request Headers: $headers");

      if (response.statusCode == 200) {
        final decoded = json.decode(response.body);
        final List<dynamic> data = decoded['data'];

        return data.map((json) => Doctor.fromJson(json)).toList();
      } else {
        throw ('Failed to load doctor list');
      }
    } on SocketException {
      Fluttertoast.showToast(
        msg: "Network error! Please retry.",
        toastLength: Toast.LENGTH_SHORT,
        gravity: ToastGravity.BOTTOM,
        backgroundColor: Colors.red,
        textColor: Colors.white,
      );
      throw Exception('Network error: Unable to fetch');
    } catch (e) {
      print("Error: $e");
      throw Exception('An error occurred: $e');
    }
  }
  @override
  Widget build(BuildContext context) {
    return
      PopScope(
        canPop: false, // prevent default back
        onPopInvoked: (didPop) async {
          if (!didPop) {
            // Replace current screen with another one
            Navigator.pushReplacement(
              context,
              MaterialPageRoute(builder: (context) => AdminDashboardScreen()),
            );
          }
        },


      child: Scaffold(
        appBar: AppBar(
          backgroundColor:Color(0xffF2903E),
          leading: IconButton(
            icon: Icon(
              Icons.arrow_back_ios_new_outlined,
              color: Colors.white,
            ),
            onPressed: () {
              Navigator.of(context).pushReplacement(
                MaterialPageRoute(
                  builder: (context) => AdminDashboardScreen(),
                ),
              );
            },
          ),
          title: Text('Doctors List', style: TextStyle(color: Colors.white)),
        ),
        body: Stack(
          children: [
            Container(
              height: 80,
              decoration: BoxDecoration(
                color:Color(0xffF2903E), // Example background color
                borderRadius: BorderRadius.only(
                  bottomLeft: Radius.circular(20.0),
                  bottomRight: Radius.circular(20.0),
                ),
              ), // Example background color
            ),
            Padding(
              padding: const EdgeInsets.only(left: 5.0, right: 10.0, top: 20.0, bottom: 0.0),
              child: FutureBuilder<List<Doctor>>(
                future: _doctorFuture, // Using the initialized _doctorFuture
                builder: (context, snapshot) {
                  if (snapshot.connectionState == ConnectionState.waiting) {
                    return Center(child: CircularProgressIndicator());
                  } else if (snapshot.hasError) {
                    return Center(child: Text('No doctors found',style: TextStyle(fontWeight: FontWeight.bold, color: Colors.black,
                      fontSize: 20,),));
                  } else if (!snapshot.hasData || snapshot.data!.isEmpty) {
                    return Center(child: Text('No doctors found'));
                  }

                  final doctors = snapshot.data!;

                  return ListView.builder(
                    itemCount: doctors.length,
                    itemBuilder: (context, index) {
                      final doctor = doctors[index];

                      return GestureDetector(
                        onTap: () {
                          Navigator.of(context).pushReplacement(MaterialPageRoute(
                            builder: (context) => HospitalSingleDoctorProfileScreen( DoctorID:doctor.id.toString(),DoctorName:doctor.name.toString()),
                          ));
                          print("Card tapped!");
                        },
                        child: Container(
                          margin: EdgeInsets.symmetric(vertical: 6, horizontal: 8),
                          padding: EdgeInsets.all(12),
                          decoration: BoxDecoration(
                            color: Colors.white,
                            borderRadius: BorderRadius.circular(12),
                            boxShadow: [
                              BoxShadow(
                                color: Colors.grey.shade200,
                                blurRadius: 5,
                                offset: Offset(0, 2),
                              ),
                            ],
                          ),
                          child: Row(
                            crossAxisAlignment: CrossAxisAlignment.start,
                            children: [
                              // Image (null check)
                              ClipRRect(
                                borderRadius: BorderRadius.circular(12),
                                child: doctor.profile != null && doctor.profile!.isNotEmpty
                                    ? Image.network(
                                  doctor.profile!,
                                  width: 100,
                                  height: 100,
                                  fit: BoxFit.cover,
                                  errorBuilder: (context, error, stackTrace) {
                                    return Container(
                                      width: 100,
                                      height: 100,
                                      color: Colors.grey.shade200,
                                      child: Icon(Icons.person, size: 50, color: Colors.grey),
                                    );
                                  },
                                )
                                    : Container(
                                  width: 100,
                                  height: 100,
                                  color: Colors.grey.shade200,
                                  child: Icon(Icons.person, size: 50, color: Colors.grey),
                                ),
                              ),

                              const SizedBox(width: 12),
                              // Details
                              Expanded(
                                child: Column(
                                  crossAxisAlignment: CrossAxisAlignment.start,
                                  children: [
                                    Row(
                                      children: [
                                        Expanded(
                                          child: Text(
                                            doctor.name.toString(),
                                            overflow: TextOverflow.ellipsis,
                                            style: TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
                                          ),
                                        ),
                                        GestureDetector(
                                          onTap: () {
                                            print("Icon tapped!");
                                          },
                                          child: Icon(Icons.person, size: 30),
                                        ),
                                      ],
                                    ),
                                    SizedBox(height: 4),
                                    Text(" ${doctor.roleName}",style: TextStyle(color:Color(0xff004D40),fontWeight: FontWeight.bold ),),
                                    Text("${doctor.experiance} years Experience "),
                                    SizedBox(height: 8),
                                    Wrap(
                                      crossAxisAlignment: WrapCrossAlignment.center,
                                      spacing: 10,
                                      children: [
                                        Row(
                                          mainAxisSize: MainAxisSize.min,
                                          children: [
                                            Icon(Icons.volume_up, size: 15,color:Color(0xff004D40) ,),
                                            SizedBox(width: 4),
                                            Text("${doctor.pendingList}",style: TextStyle(fontWeight: FontWeight.bold)),
                                            SizedBox(width: 10),
                                            Icon(Icons.download, size: 15,color:Color(0xff004D40)),
                                            SizedBox(width: 4),
                                            Text("${doctor.uploadList}",style: TextStyle(fontWeight: FontWeight.bold),),
                                          ],
                                        ),
                                        Align(
                                          alignment: Alignment.topRight,
                                          child: ElevatedButton(
                                            child: Text('Files', style: TextStyle(color: Colors.white)),
                                            style: ElevatedButton.styleFrom(
                                              backgroundColor: Colors.orange,
                                              shape: RoundedRectangleBorder(
                                                borderRadius: BorderRadius.circular(5.0),
                                              ),
                                            ),
                                            onPressed: () {
                                              Navigator.of(context).pushReplacement(MaterialPageRoute(
                                                builder: (context) => SelectedDoctorFileScreen( DoctorID:doctor.id.toString(),DoctorName:doctor.name.toString()),
                                              ));
                                              // Handle doctor files
                                            },
                                          ),
                                        ),
                                      ],
                                    ),
                                  ],
                                ),
                              ),
                            ],
                          ),
                        ),
                      );
                    },
                  );
                },
              ),
            ),
          ],
        ),
      ),
    );
  }
}

class Doctor {
  Doctor({
    required this.id,
    required this.name,
    required this.email,
    required this.profile,
    required this.roleName,
    required this.employeeId,
    required this.status,
    required this.city,
    required this.emailVerifiedAt,
    required this.password,
    required this.rememberToken,
    required this.hospital,
    required this.experiance,
    required this.mobile,
    required this.authLevel,
    required this.createdAt,
    required this.updatedAt,
    required this.uploadList,
    required this.pendingList,
  });

  final int? id;
  final String? name;
  final String? email;
  final String? profile;
  final String? roleName;
  final String? employeeId;
  final int? status;
  final dynamic city;
  final dynamic emailVerifiedAt;
  final String? password;
  final dynamic rememberToken;
  final int? hospital;
  final int? experiance;
  final String? mobile;
  final int? authLevel;
  final DateTime? createdAt;
  final DateTime? updatedAt;
  final String? uploadList;
  final String? pendingList;

  factory Doctor.fromJson(Map<String, dynamic> json){
    return Doctor(
      id: json["id"],
      name: json["name"],
      email: json["email"],
      profile: json["profile"],
      roleName: json["role_name"],
      employeeId: json["employee_id"],
      status: json["status"],
      city: json["city"],
      emailVerifiedAt: json["email_verified_at"],
      password: json["password"],
      rememberToken: json["remember_token"],
      hospital: json["hospital"],
      experiance: json["experiance"],
      mobile: json["mobile"],
      authLevel: json["auth_level"],
      createdAt: DateTime.tryParse(json["created_at"] ?? ""),
      updatedAt: DateTime.tryParse(json["updated_at"] ?? ""),
      uploadList: json["upload_list"],
      pendingList: json["pending_list"],
    );
  }

}